/*
 * This file is a part of IDDCoreFoundation framework.
 * Copyright (C) 1997-2007 Klajd(Clyde) M. Deda
 *
 * http://www.id-design.com/software/index.php
 * Free to use at your own risk.
 */


/************
** IMPORTS **
************/
#import "IDDLog.h"
#import "NSFileManager_IDDCoreFoundation.h"
#import "NSThread_IDDCoreFoundation.h"
#include <objc/objc-class.h>

/************
** STATICS **
************/
static IDDLog*  _standardLog = nil;
static NSString*  IDDLOG_INFO_CHAR  = @"I";
static NSString*  IDDLOG_DEBUG_CHAR = @"D";
static NSString*  IDDLOG_TRACE_CHAR = @"T";
static NSString*  IDDLOG_ERROR_CHAR = @"E";
static const int  IDDLOG_INFO = 1;
static const int  IDDLOG_DEBUG = 2;
static const int  IDDLOG_TRACE = 4;
static const int  IDDLOG_ERROR = 8;
static char  IDDLOG_CHAR[16] = "_ND_T___E";
static int  PROCESS_PID = 0;
static NSString*  PROCESS_NAME = nil;

int  IDDLOG_LEVEL = 1;
int  IDDLOG_LEVELS_FOR_ISA_MAX = 1024;
unsigned int  IDDLOG_LEVELS_FOR_ISA[1024][2];
int  IDDLOG_LEVELS_FOR_ISA_COUNT = 0;

/*******************
** IMPLEMENTATION **
*******************/
#pragma mark -
#pragma mark Public functions
IDDLog::IDDLog() {
    IDDLog(nil, NO);
}

IDDLog::IDDLog(NSString* fileName, BOOL rotateFlag) {
    int  i = 0;
	
    lock = [[NSRecursiveLock alloc] init];
    fileHandle = nil;
    lastWrite = [[NSCalendarDate date] dayOfYear];
    rotate = NO;
    baseName = nil;
    while (i<IDDLOG_LEVELS_FOR_ISA_MAX) {
        IDDLOG_LEVELS_FOR_ISA[i][0] = 0; 
        IDDLOG_LEVELS_FOR_ISA[i][1] = 0; 
        i++;
    }
	
	setLogFileName(fileName, rotateFlag);
}

IDDLog::~IDDLog() {
    [lock release];
    [fileHandle release];
    [baseName release];
}

void IDDLog::setLogFileName(NSString* aValue, BOOL rotateFlag) {
	[baseName release]; baseName = nil;
	
    if (aValue && [aValue cStringLength]) baseName = [aValue retain];
    rotate = rotateFlag;
    if ((!baseName) || (!openfile())) {
        fileHandle = [[NSFileHandle fileHandleWithStandardError] retain];
        [fileHandle writeData:[NSData dataWithBytes:"\n\n" length:2]];
        [fileHandle synchronizeFile];
        log(IDDLOG_ERROR, nil, NULL, @"IDDLog() Unable to create logfile, logging to stderr", (va_list)NULL);
    }
}

NSString* IDDLog::logFileName() {
    //    NSString*  lf = [[NSUserDefaults standardUserDefaults] stringForKey:@"LoggingFolder"];
    //    NSString*  path = [[NSUserDefaults domainPath] stringByAppendingPathComponent:lf];
    NSString*  path = @"/tmp";
    
    if (!rotate) return baseName;
    if ([[baseName pathComponents] count] > 1) {
        return [baseName stringByAppendingFormat:@".%03i", lastWrite];
    }
    if (rotate) {
        NSFileManager*  fm = [NSFileManager defaultManager];
        NSCalendarDate*  date = [NSCalendarDate calendarDate];
        
        [date dateByAddingYears:0 months:0 days:lastWrite hours:0 minutes:0 seconds:0];
        [date setCalendarFormat:@"%b_%d_%Y"];
        path = [path stringByAppendingFormat:@"/%03i.%@", lastWrite, [date description]];
        [fm createDirectoryAtPath:path attributes:nil recursive:YES];
    }
    path = [path stringByAppendingPathComponent:baseName];
    if (![[baseName pathExtension] isEqual:@"log"]) {
        path = [path stringByAppendingPathExtension:@"log"];
    }
    return path;
}

#pragma mark -
#pragma mark Class Global Functions
void IDDLog::setLevelError()  { IDDLOG_LEVEL |= IDDLOG_ERROR; }
void IDDLog::setLevelInfo()  { IDDLOG_LEVEL |= IDDLOG_INFO; }
void IDDLog::setLevelDebug()  { IDDLOG_LEVEL |= (IDDLOG_INFO | IDDLOG_DEBUG); }
void IDDLog::setLevelTrace()  { IDDLOG_LEVEL |= (IDDLOG_INFO | IDDLOG_DEBUG | IDDLOG_TRACE); }

BOOL IDDLog::errorLevel(id sender) {return (_standardLog->levelForID(sender) & IDDLOG_ERROR);}
BOOL IDDLog::infoLevel(id sender) {return (_standardLog->levelForID(sender) & IDDLOG_INFO);}
BOOL IDDLog::debugLevel(id sender) {return (_standardLog->levelForID(sender) & IDDLOG_DEBUG);}
BOOL IDDLog::traceLevel(id sender) {return (_standardLog->levelForID(sender) & IDDLOG_TRACE);}

void IDDLog::error(id sender, SEL aSelector, NSString *theEntry,...) {
    va_list  args;
    
    if (!_standardLog) IDDLog::standardLog();
    [_standardLog->lock lock];
    va_start(args, theEntry);
    _standardLog->log(IDDLOG_ERROR, sender, aSelector, theEntry, args);
    va_end(args);	
    [_standardLog->lock unlock];
}

void IDDLog::info(id sender, SEL aSelector, NSString *theEntry,...) {
    va_list  args;
    
    if (!_standardLog) IDDLog::standardLog();
    if (!(_standardLog->levelForID(sender) & IDDLOG_INFO)) return;
    [_standardLog->lock lock];
    va_start(args, theEntry);
    _standardLog->log(IDDLOG_INFO, sender, aSelector, theEntry, args);
    va_end(args);	
    [_standardLog->lock unlock];
}

void IDDLog::debug(id sender, SEL aSelector, NSString *theEntry,...) {
    va_list  args;
    
    if (!_standardLog) IDDLog::standardLog();
    if (!(_standardLog->levelForID(sender) & IDDLOG_DEBUG)) return;
    [_standardLog->lock lock];
    va_start(args, theEntry);
    _standardLog->log(IDDLOG_DEBUG, sender, aSelector, theEntry, args);
    va_end(args);	
    [_standardLog->lock unlock];
}

void IDDLog::trace(id sender, SEL aSelector, NSString *theEntry,...) {
    va_list  args;
    
    if (!_standardLog) IDDLog::standardLog();
    if (!(_standardLog->levelForID(sender) & IDDLOG_TRACE)) return;
    [_standardLog->lock lock];
    va_start(args, theEntry);
    _standardLog->log(IDDLOG_TRACE, sender, aSelector, theEntry, args);
    va_end(args);	
    [_standardLog->lock unlock];
}

IDDLog* IDDLog::standardLog() {return standardLog(nil, NO);}
IDDLog* IDDLog::standardLog(NSString* fileName, BOOL rotateFlag) {
    if (!_standardLog) {
        _standardLog = new IDDLog(fileName, rotateFlag);
        IDDLog::setLevelFromUserDefaults();
    }
    return _standardLog;
}

IDDLog* IDDLog::userLibraryLog(NSString* appName, BOOL rotateFlag) {
    NSString*  base = [@"~/Library/Logs/FAP/" stringByExpandingTildeInPath];
    NSString*  path = [base stringByAppendingPathComponent:appName];
    
    return standardLog([path stringByAppendingPathExtension:@"log"], rotateFlag);
}

void IDDLog::setLevelFromUserDefaults() {
    NSString*  value = [[NSUserDefaults standardUserDefaults] objectForKey:@"IDDLog"];
    
    if (!value) value = @"N";
    else if ([value isEqual:IDDLOG_INFO_CHAR]) setLevelInfo();
    else if ([value isEqual:IDDLOG_DEBUG_CHAR]) setLevelDebug();
    else if ([value isEqual:IDDLOG_TRACE_CHAR]) setLevelTrace();
    else if ([value isEqual:IDDLOG_ERROR_CHAR]) setLevelError();
}

void IDDLog::loadLevelsFromArgumentLine(NSString* aValue) {
    NSString*  logMessage = nil;
    
    logMessage = [NSString stringWithFormat:@"IDDLog::loadLevelsFromArgumentLine(%@)", aValue];
    _standardLog->log(IDDLOG_INFO, nil, NULL, logMessage, (va_list)NULL);
    if ([aValue length]) {
        NSMutableArray*  tokens = [NSMutableArray arrayWithArray:[aValue componentsSeparatedByString:@" "]];
        NSDictionary*  _args = [[NSUserDefaults standardUserDefaults] volatileDomainForName:@"NSArgumentDomain"];
        NSMutableDictionary*  args = [NSMutableDictionary dictionaryWithDictionary:_args];
        int  argsNeedUpdate = 0;
        
        [tokens removeObject:@""];
        [tokens removeObject:@" "];
        for (int i=0; i<[tokens count]; i++) {
            NSString*  key = [tokens objectAtIndex:i];
            
            if ([key hasPrefix:@"-"] && (i < [tokens count])) {
                if (![args objectForKey:[tokens objectAtIndex:i+1]]) {
                    NSString*  argumentKey = [key substringFromIndex:1];
                    id  defaultValue = [[NSUserDefaults standardUserDefaults] objectForKey:argumentKey];
                    
                    if (![defaultValue isEqual:[tokens objectAtIndex:i+1]]) {
                        [args setObject:[tokens objectAtIndex:i+1] forKey:argumentKey];
                        argsNeedUpdate ++;
                    }
                }
            }
        }
        if (argsNeedUpdate > 0) {
            [[NSUserDefaults standardUserDefaults] removeVolatileDomainForName:NSArgumentDomain];
            [[NSUserDefaults standardUserDefaults] setVolatileDomain:args forName:NSArgumentDomain];
            [NSUserDefaults resetStandardUserDefaults];
            logMessage = [NSString stringWithFormat:@"IDDLog::loadLevelsFromArgumentLine(%@)", args];
            _standardLog->log(IDDLOG_INFO, nil, NULL, logMessage, (va_list)NULL);
            // [[NSUserDefaults standardUserDefaults] synchronize];
        }
        setLevelFromUserDefaults();
    }
}

#pragma mark -
#pragma mark Private functions
BOOL IDDLog::openfile() {
    BOOL  didSucceed = YES;
    
    @try {
        NSFileHandle*  newFileHandle = [NSFileHandle fileHandleForWritingAtPath:logFileName()];
        
        // attempt to create the file if it doesn't already exist
        if (!newFileHandle) {
            NSFileManager*  fm = [NSFileManager defaultManager];
            NSDictionary*  attributes = [NSDictionary dictionaryWithObject:[NSNumber numberWithInt:0755] forKey:@"NSFilePosixPermissions"];
            // TO-DO: Set file permissions to a proper value
            if ([fm createDirectoryAtPath:[logFileName() stringByDeletingLastPathComponent] attributes:attributes recursive:YES]) {
                attributes = [NSDictionary dictionaryWithObject:[NSNumber numberWithInt:0644] forKey:@"NSFilePosixPermissions"];
                if ([fm createFileAtPath:logFileName() contents:nil attributes:attributes]) {
                    newFileHandle = [NSFileHandle fileHandleForWritingAtPath:logFileName()];
                }
            }
        }
        
        if (!newFileHandle) {
            NSLog(@"IDDLog could not open %@", logFileName());
            didSucceed = NO;
        } else {
            NSLog(@"IDDLog will start logging to file %@", logFileName());
            [newFileHandle seekToEndOfFile];
            [fileHandle release];
            fileHandle = [newFileHandle retain];
            [fileHandle writeData:[NSData dataWithBytes:"\n\n" length:2]];
            [fileHandle synchronizeFile];
        }
    } @catch (NSException* exception) {
        NSLog(@"IDDLog::openfile() Error %@ while opening %@", [exception reason], logFileName());
        didSucceed = NO;
    }
    return didSucceed;
}

int IDDLog::logLevelForNSString(NSString* _this) {
    int  logLevel = IDDLOG_LEVEL;
    NSString*  value = [[NSUserDefaults standardUserDefaults] stringForKey:_this];
    
    if (value) {        
        if ([value isEqual:IDDLOG_INFO_CHAR]) logLevel = IDDLOG_INFO;
        else if ([value isEqual:IDDLOG_DEBUG_CHAR]) logLevel |= (IDDLOG_INFO | IDDLOG_DEBUG);
        else if ([value isEqual:IDDLOG_TRACE_CHAR]) logLevel |= (IDDLOG_INFO | IDDLOG_DEBUG | IDDLOG_TRACE);
        else if ([value isEqual:IDDLOG_ERROR_CHAR]) logLevel |= IDDLOG_ERROR;
    } else {
        NSString*  entry = [NSString stringWithFormat:@"logLevelForNSString() add -%@ [N|E|D|T] to change the log level, using %c", _this, IDDLOG_CHAR[logLevel]];
        log(IDDLOG_INFO, nil, NULL, entry, (va_list)NULL);
    }
    return logLevel;
}

int IDDLog::isNSStringClass(struct objc_object* _this) {
    static int  specialCount = 3;
    static struct objc_class*  special[3] = {
        NSClassFromString(@"NSString"), 
        NSClassFromString(@"NSCFString"), 
        NSClassFromString(@"NSConstantString")};
    int  i;
    
    for (i=0; i<specialCount; i++) {if (_this->isa == special[i]) return 1;}
    return 0;
}

int IDDLog::levelForID(NSObject* _this) {
    int  logLevel = IDDLOG_LEVEL;
    
    if (_this) {
        BOOL  levelFound = NO;
        int  i = 0;
		
		if (isNSStringClass(_this)) {            
            // strings
			for (i=0; i<IDDLOG_LEVELS_FOR_ISA_COUNT; i++) {
				if (IDDLOG_LEVELS_FOR_ISA[i][0] == (unsigned int)_this) {
					logLevel = IDDLOG_LEVELS_FOR_ISA[i][1];
                    levelFound = YES;
                    break;
				}
			}

			if (!levelFound && (IDDLOG_LEVELS_FOR_ISA_COUNT < IDDLOG_LEVELS_FOR_ISA_MAX)) {
                logLevel = logLevelForNSString((NSString*)_this);
				IDDLOG_LEVELS_FOR_ISA[IDDLOG_LEVELS_FOR_ISA_COUNT][0] = (unsigned int)_this;
				IDDLOG_LEVELS_FOR_ISA[IDDLOG_LEVELS_FOR_ISA_COUNT][1] = logLevel;
				IDDLOG_LEVELS_FOR_ISA_COUNT ++;
			}
		} else {
            // real classes
            struct objc_class*  __this = _this->isa;
            
            //if (strcmp(_this->isa->name, "FAServer") == 0) {
            //    NSLog(@"crap");
            //}
			for (i=0; i<IDDLOG_LEVELS_FOR_ISA_COUNT; i++) {
				if (IDDLOG_LEVELS_FOR_ISA[i][0] == (unsigned int)__this) {
					logLevel = IDDLOG_LEVELS_FOR_ISA[i][1];
                    levelFound = YES;
                    break;
				}
			}
			if (!levelFound && (IDDLOG_LEVELS_FOR_ISA_COUNT < IDDLOG_LEVELS_FOR_ISA_MAX)) {
                logLevel = logLevelForNSString([_this className]);
				IDDLOG_LEVELS_FOR_ISA[IDDLOG_LEVELS_FOR_ISA_COUNT][0] = (unsigned int)__this;
				IDDLOG_LEVELS_FOR_ISA[IDDLOG_LEVELS_FOR_ISA_COUNT][1] = logLevel;
				IDDLOG_LEVELS_FOR_ISA_COUNT ++;
			}
		}		
    }
    return logLevel;
}

void IDDLog::log(int aLevel, id sender, SEL aSelector, NSString *theEntry, va_list args) {
    @try {
        NSCalendarDate*  theDate = [[NSCalendarDate alloc] init];
        NSString*  theString = [[NSString alloc] initWithFormat:theEntry arguments:args];
        NSString*  theLogEntry = nil;
        NSData*  theData = nil;
        NSThread*  currentThread = [NSThread currentThread];
        NSString*  stamp = nil;
        
        if (rotate) {
            int  curJulian = [theDate dayOfYear];
            
            // Check for new day
            if (curJulian != lastWrite) {
                [fileHandle closeFile];
                lastWrite = curJulian;
                _standardLog->openfile();
            }
        }
        
        [theDate setCalendarFormat:@"%Y-%m-%d %H:%M:%S.%F"];
        if (!PROCESS_PID) PROCESS_PID = [[NSProcessInfo processInfo] processIdentifier];
        if (!PROCESS_NAME) PROCESS_NAME = [[NSProcessInfo processInfo] processName];

        // date time.milliseconds app[pid.thread] logLevel
        // stamp = [[NSString alloc] initWithFormat:@"%@ %@[%d.%x] %c", theDate, PROCESS_NAME, PROCESS_PID, (unsigned int)currentThread, IDDLOG_CHAR[aLevel]];
        stamp = [[NSString alloc] initWithFormat:@"%@ %@[%d.T%04d] %c", theDate, PROCESS_NAME, PROCESS_PID, NSThread_threadIndex(currentThread), IDDLOG_CHAR[aLevel]];
        // pid.time.milliseconds.logLevel.thread.threadid
        // stamp = [[NSString alloc] initWithFormat:@"%d.%@.%c.0x%x.T%04d", PROCESS_PID, theDate, IDDLOG_CHAR[aLevel], currentThread, NSThread_threadIndex(currentThread)];
            
        if (!sender || !aSelector) {
            theLogEntry = [[NSString alloc] initWithFormat:@"%@ ::%@\n", stamp, theString];
        } else {
            if (isNSStringClass(sender)) {
                theLogEntry = [[NSString alloc] initWithFormat:@"%@ [%@(0x%x) %s] %@\n", stamp, sender, (int)sender, sel_getName(aSelector), theString];
            } else {
                theLogEntry = [[NSString alloc] initWithFormat:@"%@ [%s(0x%x) %s] %@\n", stamp, sender->isa->name, (int)sender, sel_getName(aSelector), theString];
            }
        }
        
        theData = [[NSData alloc] initWithBytes:[theLogEntry cString] length:[theLogEntry cStringLength]];
        // NSLog(@"%d bytes, %@", [theData length], theLogEntry);
        [fileHandle writeData:theData];
        [fileHandle synchronizeFile];
        
        [theData release];
        [theLogEntry release];
        [stamp release];
        [theString release];
        [theDate release];
    } @catch (NSException* exception) {
        NSLog(@"IDDLog::log() Exception:\n");
        NSLog(@"IDDLog::log() Reason: [%s]\n", [[exception description] cString]);
    }
}

